
/*
  commands:

  0 -  39 : shift 1 - 40 bits w/TMS                                           <data>
 40 -  79 : shift 1 - 40 bits wo/TMS                                          <data>
 80 - 119 : shift 1 - 40 bits w/TMS and read                                  <data>
120 - 159 : shift 1 - 40 bits wo/TMS and read                                 <data>
160 - 221 : shift 1 - 5 inline bits w/TMS
222       : shift TDI and TMS 1-255 pairs                                     <count8>, <data>
223       : shift TDI and TMS 1-255 pairs and read                            <count8>, <data>
224       : shift 1-255 zeros w/TMS                                           <count8>
225       : shift 1-255 ones w/TMS                                            <count8>
226       : shift 1-255 zeros wo/TMS                                          <count8>
227       : shift 1-255 ones wo/TMS                                           <count8>
228       : shift 1-255 zeros w/TMS and read                                  <count8>
229       : shift 1-255 ones w/TMS and read                                   <count8>
230       : shift 1-255 zeros wo/TMS and read                                 <count8>
231       : shift 1-255 ones wo/TMS and read                                  <count8>
232 - 239 : send TDI/TMS and read TDO (1 bit) if bit 2 set
240 - 255 : shift 4 bits w/TMS

*/

typedef unsigned char byte_t;

byte_t* jtag_recv_buf;
byte_t jtag_recv_buf_pos, jtag_recv_buf_fractional_pos;


void jtag_write_bit_to_recv_buf(byte_t bit) {
  if( bit )
    jtag_recv_buf[jtag_recv_buf_pos] |=  jtag_recv_buf_fractional_pos;
  else
    jtag_recv_buf[jtag_recv_buf_pos] &= ~jtag_recv_buf_fractional_pos;
  jtag_recv_buf_fractional_pos <<= 1;
  if( !jtag_recv_buf_fractional_pos ) {
    jtag_recv_buf_fractional_pos = 1;
    ++jtag_recv_buf_pos;
  }
}
void jtag_write_byte_to_recv_buf(byte_t byte) {
  if( jtag_recv_buf_fractional_pos == 1 ) {
    jtag_recv_buf[jtag_recv_buf_pos++] = byte;
  } else {
    byte_t byte2, temp;
    temp = jtag_recv_buf_fractional_pos;
    byte2 = byte;
    while( temp != 1 ) {
      byte <<= 1;
      byte2 >>= 1;
      temp >>= 1;
    }
    jtag_recv_buf[jtag_recv_buf_pos] = (jtag_recv_buf[jtag_recv_buf_pos]&(jtag_recv_buf_fractional_pos-1))|byte;
    ++jtag_recv_buf_pos;
    jtag_recv_buf[jtag_recv_buf_pos] = byte2;
  }
}

void jtag_shift_bits(byte_t* ptr, byte_t count, byte_t read, byte_t finish, byte_t inc) {
  byte_t bit = 1;
  while( count ) {
    if( (*ptr) & bit )
      TDI_HI();
    else
      TDI_LO();
    if( count == 1 && finish )
      TMS_HI();
    delay();
    TCK_HI();
    delay();
    if( read )
      jtag_write_bit_to_recv_buf(TDO_RD());
    TCK_LO();
    --count;
    if( inc ) {
      bit <<= 1;
      if( !bit ) {
        ++ptr;
        bit = 1;
      }
    }
  }
  TMS_LO();
  TDI_LO();
}

void jtag_shift_tms(byte_t data, byte_t bits) {
  while( bits ) {
    if( data&1 )
      TMS_HI();
    else
      TMS_LO();
    delay();
    TCK_HI();
    delay();
    TCK_LO();
    data >>= 1;
    --bits;
  }
  TMS_LO();
}

void jtag_do_tdi_tms_pairs(byte_t* ptr, byte_t count, byte_t read) {
  byte_t bit = 1;

  while( count ) {
    if( (*ptr) & bit )
      TDI_HI();
    else
      TDI_LO();
    bit <<= 1;
    if( (*ptr) & bit )
      TMS_HI();
    else
      TMS_LO();
    delay();
    TCK_HI();
    delay();
    if( read )
      jtag_write_bit_to_recv_buf(TDO_RD());
    TCK_LO();
    --count;
    bit <<= 1;
    if( !bit ) {
      ++ptr;
      bit = 1;
    }
  }
  TMS_LO();
  TDI_LO();
}

// rest state - TDO, TMS and TCK low
byte_t do_jtag(byte_t* in_buf, byte_t in_bytes, byte_t* out_buf) {
  byte_t pos, cmd, count, *ptr;

  jtag_recv_buf = out_buf;
  jtag_recv_buf_pos = 0;
  jtag_recv_buf_fractional_pos = 1;

  pos = 0;
  while( pos < in_bytes ) {
    cmd = in_buf[pos];
    ++pos;
//    if( pos == in_bytes )
//      break; // not enough data in buffer to complete command

    if( cmd < 222 ) {
      if( cmd < 160 ) {
        // shift bits
        byte_t finish = 1, read = 0;
        if( cmd >= 80 ) {
          read = 1;
          cmd -= 80;
        }
        if( cmd >= 40 ) {
          finish = 0;
          cmd -= 40;
        }
        count = cmd+1;
        ptr = in_buf + pos;
        pos += ((count+7)>>3);
        if( pos > in_bytes )
          break; // not enough data in buffer to complete command
        jtag_shift_bits(ptr, count, read, finish, 1);
      } else {
        // shift tms bits
        byte_t data, bits, val;
        data = cmd - 160;
        bits = 1;
        val = 2;
        while( data >= val ) {
          ++bits;
          data -= val;
          val <<= 1;
        }
        jtag_shift_tms(data, bits);
      }
    } else if( cmd < 232 ) {
      if( pos == in_bytes )
        break; // not enough data in buffer to complete command
      count = in_buf[pos];
      ++pos;

      if( cmd < 224 ) {
        // shift tdi/tms pairs
        byte_t* ptr = in_buf + pos;
        pos += ((count+3)>>2);
        if( pos > in_bytes )
          break; // not enough data in buffer to complete command
        jtag_do_tdi_tms_pairs(ptr, count, cmd&1);
      } else {
        // shift zeros/ones
        jtag_shift_bits(&cmd, count, (cmd&4), !(cmd&2), 0);
      }
    } else {
      if( cmd < 240 ) {
        // one tdi/tms pair
        jtag_do_tdi_tms_pairs(&cmd, 1, (cmd & 4));
      } else {
        // shift four bits w/TMS
        jtag_shift_bits(&cmd, 4, 0, 1, 1);
      }
    }
  }

  // return number of bytes to send back to host
  return jtag_recv_buf_pos + (jtag_recv_buf_fractional_pos == 1 ? 0 : 1);
}
